///////////////////////////////////////////////////////////////////////////////
// (C) Copyright 2001 by Autodesk, Inc. 
//
// Permission to use, copy, modify, and distribute this software in
// object code form for any purpose and without fee is hereby granted, 
// provided that the above copyright notice appears in all copies and 
// that both that copyright notice and the limited warranty and
// restricted rights notice below appear in all supporting 
// documentation.
//
// AUTODESK PROVIDES THIS PROGRAM "AS IS" AND WITH ALL FAULTS. 
// AUTODESK SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTY OF
// MERCHANTABILITY OR FITNESS FOR A PARTICULAR USE.  AUTODESK, INC. 
// DOES NOT WARRANT THAT THE OPERATION OF THE PROGRAM WILL BE
// UNINTERRUPTED OR ERROR FREE.
//
// Use, duplication, or disclosure by the U.S. Government is subject to 
// restrictions set forth in FAR 52.227-19 (Commercial Computer
// Software - Restricted Rights) and DFAR 252.227-7013(c)(1)(ii)
// (Rights in Technical Data and Computer Software), as applicable.
//
// CREATED BY:
//	Hugues Wisniewski, June 2001
//
// DESCRIPTION:
// 	Implementation of namespace AcMapOrclApiTestTools

#include "stdafx.h"
#include "AcMapOrclApiTestTools.h"
#include "AcMapOracleConnection.h"
#include "AcMapOrclApiTestInformationDlg.h"
#include "winbase.h"

//******************************************************************************************
BOOL AcMapOrclApiTestTools::IsDotDirectory(LPWIN32_FIND_DATA pData)
{
    BOOL retval = FALSE;
    if (pData && (pData->dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY))
    {
        if (pData->cFileName[0] == '.')
        {
            if ( pData->cFileName[1] == '\0' ||
                (pData->cFileName[1] == '.' &&
                 pData->cFileName[2] == '\0'))
            {
                retval = TRUE;
            }
        }
    }
    return retval;
}

//******************************************************************************************
void AcMapOrclApiTestTools::CheckEndSlash(CString& cszDirectory)
{
    int nLength = cszDirectory.GetLength();
 
    if( nLength > 0  && cszDirectory.Right( 1 ) != _T("\\"))
    {
        cszDirectory += _T("\\");
    }
}

#define DISK_A 0x00000001L
#define DISK_B 0x00000002L
#define DISK_C 0x00000004L
#define DISK_D 0x00000008L
#define DISK_E 0x00000010L
#define DISK_F 0x00000020L
#define DISK_G 0x00000040L
#define DISK_H 0x00000080L
#define DISK_I 0x00000100L
#define DISK_J 0x00000200L


//******************************************************************************************
// looks for the file anywhere on any of the local drives, in the order c,d,e,f,g,h,i,j

BOOL AcMapOrclApiTestTools::FindAndOpenSqlWhereClauseSamplesFile(
	CStdioFile& whereClauseSamplesFile,
	LPCTSTR	szFileName,
	BOOL bIncludeSubDirs)
{
	DWORD vectDisk[]={DISK_C, DISK_D, DISK_E, DISK_F, DISK_G, DISK_H, DISK_I, DISK_J};
	DWORD dwAllDisks=GetLogicalDrives();

	//first look on local drive
	if (AcMapOrclApiTestTools::FindAndOpenSqlWhereClauseSamplesFile(
		whereClauseSamplesFile,
		"\\", 
		szFileName,
		bIncludeSubDirs))
	{
		return true;
	}

	//then on other disks
	for (int i=0; i<8; i++)
	{
		if (dwAllDisks & vectDisk[i])
		{
			CString strBaseDirectory=(char)((int)'C'+i);
			strBaseDirectory+=":\\";

			if (AcMapOrclApiTestTools::FindAndOpenSqlWhereClauseSamplesFile(
				whereClauseSamplesFile,
				strBaseDirectory, 
				szFileName,
				bIncludeSubDirs))
			{
				return true;
			}
		}
	}

	return false;
}

//******************************************************************************************
// This function looks for the file, starting in the directory "cszBaseDirectory"
// and opens it when found. If not found retaurns false

BOOL AcMapOrclApiTestTools::FindAndOpenSqlWhereClauseSamplesFile(
	CStdioFile& whereClauseSamplesFile,
	const CString& cszBaseDirectory, 
	LPCTSTR	szFileName,
	BOOL bIncludeSubDirs)
{
    WIN32_FIND_DATA     findData;
    CString             cszDir = cszBaseDirectory;
 
    CheckEndSlash( cszDir );
 
    if( bIncludeSubDirs )
    {
        CString  cszAllMask = cszDir + _T("*.*");
 
        HANDLE  hFindAll = FindFirstFile( (LPCTSTR)cszAllMask, &findData ); 
        if( hFindAll != INVALID_HANDLE_VALUE )
        {
            do
            {
				if((findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY)
				   && !IsDotDirectory(&findData))
				{
					CString  cszSubSearchDir = cszDir + findData.cFileName;
					if (FindAndOpenSqlWhereClauseSamplesFile(
						whereClauseSamplesFile, 
						cszSubSearchDir, 
						szFileName, 
						bIncludeSubDirs))
					{
						FindClose(hFindAll);
						return TRUE;
					}
				}
            } while (FindNextFile(hFindAll, &findData));
            FindClose(hFindAll);
        }
    }
 
    CString  cszFileMask = cszDir + szFileName;
 
    HANDLE  hFindFiles = FindFirstFile((LPCTSTR)cszFileMask, &findData);
    if(hFindFiles != INVALID_HANDLE_VALUE)
    {
		do
		{
			if( (findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0 )
			{
				if (!whereClauseSamplesFile.Open((const char*)cszFileMask, CFile::modeRead | CFile::typeText))
				{
					return false;
				}

				FindClose(hFindFiles);
				return TRUE;
			}
		} while (FindNextFile( hFindFiles, &findData) );
		FindClose( hFindFiles );
    }

	return FALSE;
}

//******************************************************************************************
// For a given table, fills the two vctors with the different column names and for each name, 
// the corresponding data type.
// There is a one to one corresponding value in the two vectors
// At a given index we have a pair (ColumnName, DataType) in the vectors, something like
// ("ENTITYTYPE","VARCHAR2")
// A std::map is not used on purpose because of crossing boundary problems beween DLLs
// Returns false if an error occured

bool AcMapOrclApiTestTools::FillColumnNamesDataTypes(
	const std::string& strTableName, //input table name
	std::vector<std::string>& vectColumnNames, //output vector of all the different column names
	std::vector<std::string>& vectDataTypes, //output vector of data types corresponding to column names in vectColumnNames
	bool bUpperCase)//true to have all the names in upper case
{
	const char *kpcSql="select COLUMN_NAME, DATA_TYPE, DATA_SCALE, DATA_PRECISION "
		"from SYS.ALL_TAB_COLUMNS "
		"where TABLE_NAME='%s' "
		"and OWNER='%s'";
	char szColumnName[256];
	char szDataType[256];
	std::string strSchemaBuffer=AcMapOracleGetConnection()->UserName();
	std::string strTableNameBuffer=strTableName;

	_tcsupr(const_cast<char *>(strTableNameBuffer.c_str()));
	_tcsupr(const_cast<char*>(strSchemaBuffer.c_str()));

	//form the sql string
	CString strSql;
	strSql.Format(
		kpcSql,
		strTableNameBuffer.c_str(),
		strSchemaBuffer.c_str());

	vectColumnNames.clear();
	vectDataTypes.clear();

	ODynaset ds(
		AcMapOracleGetConnection()->Database(),
		(const char*)strSql, 
		ODYNASET_NOCACHE|ODYNASET_READONLY|ODYNASET_NO_MOVEFIRST);

	for (ds.MoveFirst();!ds.IsEOF();ds.MoveNext())
	{
		if (OSUCCESS==ds.GetFieldValue("COLUMN_NAME", szColumnName,256)
			&& OSUCCESS ==ds.GetFieldValue("DATA_TYPE", szDataType,256))
		{
			if (bUpperCase)
			{
				_tcsupr(szColumnName);
				_tcsupr(szDataType);
			}

			// In Oracle if we have a column of type INTEGER
			// when we ask for the data type, NUMBER is returned but
			// with a scale of 0 and precision=NULL (beware this is not 0, it is no value)
			// In that case we put the type INTEGER in the vector
			// We also have an INTEGER if the type of the column
			// was NUMBER(size). In this case 
			// scale = 0 and precision = size

			if (0==strcmp(szDataType, "NUMBER"))
			{
				OValue valDataScale, valPrecision;

				if (OSUCCESS==ds.GetFieldValue("DATA_SCALE", &valDataScale)
					&& OSUCCESS==ds.GetFieldValue("DATA_PRECISION", &valPrecision))
				{
					double dfDataScale, dfPrecision;
					oboolean bDataScaleIsNull=valDataScale.IsNull();
					oboolean bPrecisionIsNull=valPrecision.IsNull();

					// If a column was created with type INTEGER, that is integer C type
					// we have DATA_SCALE=0 and DATA_PRECISION=NULL
					// If a column was created with type NUMBER, that is double C type
					// we have DATA_SCALE=NULL and DATA_PRECISION=NULL
					// If a column was created with type NUMBER(14), that is integer C type
					// we have DATA_SCALE=0 and DATA_PRECISION=14
					// If a column was created with type NUMBER(14,2), that is double C type
					// we have DATA_SCALE=2 and DATA_PRECISION=14

					if (!bDataScaleIsNull
						&& OSUCCESS==ds.GetFieldValue("DATA_SCALE", &dfDataScale)
						&& 0.==dfDataScale
						&& OSUCCESS==ds.GetFieldValue("DATA_PRECISION", &dfPrecision))
					{
						if (bPrecisionIsNull
							|| (!bPrecisionIsNull && dfPrecision>0.))
						{
							strcpy(szDataType, "INTEGER");
						}
					}

				}
			}
			vectColumnNames.push_back(szColumnName);
			vectDataTypes.push_back(szDataType);
		}
		else
		{
			AfxMessageBox("Failure reading table column names");
			vectColumnNames.clear();
			vectDataTypes.clear();
			return false;
		}
	}

	return vectColumnNames.size()>0?true:false;
}

//******************************************************************************************
// Gets the list of string values result of a query

bool AcMapOrclApiTestTools::GetListStringValuesFromTable(
	const std::string& strTableName,
	const std::string& strQuery,
	std::vector<std::string>& vectValues) //output string values
{
	try
	{
		ODynaset dynaset(AcMapOracleGetConnection()->Database(), strQuery.c_str());
		if (!dynaset.IsOpen())
		{
			CString strMsg;
			strMsg.Format(
				"Could not read the table %s",
				strTableName.c_str());
			AfxMessageBox(strMsg);
			return false;
		}

		for(dynaset.MoveFirst(); FALSE == dynaset.IsEOF(); dynaset.MoveNext())
		{
			oresult ores;
			char szName[1000];

			ores = dynaset.GetFieldValue(
				0,  //only one field per record
				szName, 
				sizeof(szName));
			if (OFAILURE==ores)
			{
				CString strMsg;
				strMsg.Format(
					"Could not read the table %s",
					strTableName.c_str());
				AfxMessageBox(strMsg);
				return false;
			}

			if (strlen(szName)>0)
			{
				vectValues.push_back(szName);
			}
		}
	}
	catch(OException E)
	{
		CString strMsg;
		strMsg.Format(
			"Could not read the table %s",
			strTableName.c_str());
		AfxMessageBox(strMsg);
		return false;
	}

	return true;
}

//******************************************************************************************
// Gets the list of unique instances of string values in a column in a table

bool AcMapOrclApiTestTools::GetListStringValuesFromTableColumn(
	const std::string& strTableName,
	const std::string& strColumnName,
	std::vector<std::string>& vectValues) //output string values
{
	CString strQuery;
	
	strQuery.Format(
		"select Table0.%s "
		"from %s Table0 "
		"group by Table0.%s",
		strColumnName.c_str(),
		strTableName.c_str(),
		strColumnName.c_str());

	return GetListStringValuesFromTable(
		strTableName,
		(const char*)strQuery,
		vectValues);
}

//******************************************************************************************
bool AcMapOrclApiTestTools::ReadAllSampleSqlWhereClauses(
	std::vector<CString>& vectSqlWhereClauses,
	std::vector<CString>& vectSqlWhereClausesTitles)
{
	CStdioFile whereClauseSamplesFile;
	CString strSqlWhereClausesFileName="SqlWhereClauses.txt";

	//we are looking for the sample that can be anywhere on the current disk
	//so we display a waiting dialog box
	AfxGetMainWnd()->BeginWaitCursor();
	AcMapOrclApiTestInformationDlg Dlg;
	Dlg.Create(IDD_API_INFORMATION);
	Dlg.ShowWindow(SW_SHOWNORMAL);
	Dlg.UpdateWindow();

	//look for the file an open it
	if (!AcMapOrclApiTestTools::FindAndOpenSqlWhereClauseSamplesFile(
		whereClauseSamplesFile,
		(const char*)strSqlWhereClausesFileName,
		true))
	{
		AfxGetMainWnd()->EndWaitCursor();
		Dlg.DestroyWindow();
		CString strMsg;
		strMsg.Format(
			"%s opening problem.\nThe file that stores the sample where clauses could not be opened.",
			(const char*)strSqlWhereClausesFileName);
		AfxMessageBox(strMsg);
		return false;
	}

	//file opened, removes the waiting dialog box
	AfxGetMainWnd()->EndWaitCursor();
	Dlg.DestroyWindow();

	CString strNext;
	CString strTitle;
	CString strWhereClause;

	//reads the text file and extract the samples
	while (whereClauseSamplesFile.ReadString(strNext))
	{
		//if end of a where clause and about to read a new one
		if (strNext.IsEmpty() || strNext=="END")
		{
			assert(!strTitle.IsEmpty());
			assert(!strWhereClause.IsEmpty());
			vectSqlWhereClausesTitles.push_back(strTitle);
			vectSqlWhereClauses.push_back(strWhereClause);

			strTitle="";
			strWhereClause="";
		}
		else
		{
			if (strNext.Left(12)=="QUERYTITLE: ")
			{
				if (strNext.GetLength()<=12)
				{
					AfxMessageBox("The title of the where clause is undefined.\nAn arbitrary value is choosen");
					strTitle.Format(
						"Undefined Sql Where Clause #%d", 
						vectSqlWhereClausesTitles.size()+1);
				}
				else
				{
					strTitle=strNext.Right(strNext.GetLength()-12);
				}
			}
			else
			{
				strWhereClause+=strNext;
				strWhereClause+="\r\n";
			}
		}
	}

	vectSqlWhereClausesTitles.push_back("Custom condition");
	vectSqlWhereClauses.push_back("");
	return true;
}
